<?php 
/*
 * Plugin Name: metals-api.com
 * Description: Official metals-api.com plugin for displaying real-time metal prices in WordPress.
 * Version: 2.7
 * Author: metals-api.com
 * Author URI: https://metals-api.com
 * Text Domain: metals-api-com
 * Requires at least: 5.0
 * Requires PHP: 7.0
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
*/ 

class MetalsAPI { 

    function saveOption($name){
        if(isset($_POST[$name])){
            update_option($name, sanitize_text_field($_POST[$name]));
        }
    }

    function saveNumericOption($name){
        if(isset($_POST[$name])){
            update_option($name, sanitize_text_field(abs(intval($_POST[$name]))));
        }
    }

    function saveHTMLOption($name){ 
        if(isset($_POST[$name])){
            update_option($name, wp_kses_post($_POST[$name]));
        }
    }

    function render_settings(){ 
        $saved = 0;
        if(isset($_POST['save'])){
            if (!isset($_POST['thenonce']) || !wp_verify_nonce($_POST['thenonce'])) {
                wp_die("ERROR: Request not valid");
            }
            $this->saveOption('metalapi_apikey'); 
            $saved = 1;
        }

        $metalapi_error = get_option('metalapi_error');

        require plugin_dir_path(__FILE__). 'templates/settings.php';
    }

    function render_info(){
        require plugin_dir_path(__FILE__). 'templates/info.php';
    }

    function render_live_price_doc(){
        require plugin_dir_path(__FILE__). 'metals_live_price.php';
    }

    function render_table_doc(){
        require plugin_dir_path(__FILE__). 'metals_table.php';
    }

    function addSettings(){
        add_menu_page('Metals API Settings', 'Metals API', 'administrator', 'metalsapi', [$this, 'render_settings']);
        add_submenu_page('metalsapi', 'API Key', 'API Key', 'administrator', 'metalsapisettings', [$this, 'render_settings']);
        add_submenu_page('metalsapi', 'Metal Live Price', 'Metal Live Price', 'administrator', 'metals_live_price', [$this, 'render_live_price_doc']);
        add_submenu_page('metalsapi', 'Multi Symbols Table', 'Multi Symbols Table', 'administrator', 'metals_table', [$this, 'render_table_doc']);
        remove_submenu_page("metalsapi", "metalsapi");
    }

    function ajax_refresh_metal_price() {
        $symbol = isset($_POST['symbol']) ? strtoupper(trim($_POST['symbol'])) : 'XAU';
        $base = isset($_POST['base']) ? strtoupper(trim($_POST['base'])) : 'USD';
        $decimals = isset($_POST['decimals']) ? intval($_POST['decimals']) : 2;
        $show_unit = isset($_POST['show_unit']) && $_POST['show_unit'] === 'true';
        $show_currency = isset($_POST['show_currency']) && $_POST['show_currency'] === 'true';
        $show_updated = isset($_POST['show_updated']) && $_POST['show_updated'] === 'true';

        $data = $this->fetch_metal_data($symbol, $base);
        
        if (!isset($data['success']) || !$data['success']) {
            wp_send_json_error(['message' => 'Error fetching data']);
            return;
        }

        // Try multiple key formats to find the rate
        $price = null;
        $rate_key = $base . $symbol;
        if (isset($data['rates'][$rate_key])) {
            $price = $data['rates'][$rate_key];
        } elseif (isset($data['rates'][$symbol])) {
            // Fallback: try symbol directly
            $price = $data['rates'][$symbol];
        } elseif (isset($data['rates']['USD' . $symbol])) {
            // Fallback: try USD prefix
            $price = $data['rates']['USD' . $symbol];
        }
        
        if ($price === null) {
            wp_send_json_error(['message' => 'Rate not found']);
            return;
        }
        $formatted_price = $this->format_price($price, $decimals);
        $metal_name = $this->get_metal_name($symbol);
        $unit = $this->get_metal_unit($symbol);
        
        $currency_symbols = [
            'USD' => '$',
            'EUR' => '€',
            'GBP' => '£',
        ];
        $currency_symbol = isset($currency_symbols[$base]) ? $currency_symbols[$base] : $base . ' ';

        ob_start();
        ?>
        <div class="metal-name"><?php echo esc_html($metal_name); ?> (<?php echo esc_html($symbol); ?>)</div>
        <div class="metal-price">
            <?php if ($show_currency): ?>
                <span class="currency-symbol"><?php echo esc_html($currency_symbol); ?></span>
            <?php endif; ?>
            <span class="price-value"><?php echo esc_html($formatted_price); ?></span>
            <?php if ($show_unit && !empty($unit)): ?>
                <span class="price-unit"> / <?php echo esc_html($unit); ?></span>
            <?php endif; ?>
        </div>
        <?php if ($show_updated && isset($data['timestamp'])): ?>
            <div class="metal-updated">Updated: <?php echo esc_html($this->format_timestamp($data['timestamp'])); ?></div>
        <?php endif; ?>
        <?php
        $html = ob_get_clean();
        wp_send_json_success($html);
    }

    function ajax_refresh_metals_table() {
        $symbols_str = isset($_POST['symbols']) ? trim($_POST['symbols']) : 'XAU,XAG,XPT,XPD';
        $symbols = array_map('trim', explode(',', $symbols_str));
        $base = isset($_POST['base']) ? strtoupper(trim($_POST['base'])) : 'USD';
        $decimals = isset($_POST['decimals']) ? intval($_POST['decimals']) : 2;
        $show_unit = isset($_POST['show_unit']) && $_POST['show_unit'] === 'true';
        $show_currency = isset($_POST['show_currency']) && $_POST['show_currency'] === 'true';
        $show_updated = isset($_POST['show_updated']) && $_POST['show_updated'] === 'true';

        $currency_symbols = [
            'USD' => '$',
            'EUR' => '€',
            'GBP' => '£',
        ];
        $currency_symbol = isset($currency_symbols[$base]) ? $currency_symbols[$base] : $base . ' ';

        $symbols_str_api = implode(',', array_map('strtoupper', $symbols));
        $data = $this->fetch_metal_data($symbols_str_api, $base);
        
        if (!isset($data['success']) || !$data['success']) {
            wp_send_json_error(['message' => 'Error fetching data']);
            return;
        }

        ob_start();
        ?>
        <table class="metals-table">
            <thead>
                <tr>
                    <th>Metal</th>
                    <th>Price</th>
                    <?php if ($show_unit): ?>
                        <th>Unit</th>
                    <?php endif; ?>
                </tr>
            </thead>
            <tbody>
                <?php
                $latest_timestamp = 0;
                foreach ($symbols as $symbol):
                    $symbol_upper = strtoupper(trim($symbol));
                    
                    // Try multiple key formats to find the rate
                    $price = null;
                    $rate_key = $base . $symbol_upper;
                    if (isset($data['rates'][$rate_key])) {
                        $price = $data['rates'][$rate_key];
                    } elseif (isset($data['rates'][$symbol_upper])) {
                        // Fallback: try symbol directly
                        $price = $data['rates'][$symbol_upper];
                    } elseif (isset($data['rates']['USD' . $symbol_upper])) {
                        // Fallback: try USD prefix
                        $price = $data['rates']['USD' . $symbol_upper];
                    }
                    
                    // Skip if price not found
                    if ($price === null) continue;
                    
                    $formatted_price = $this->format_price($price, $decimals);
                    $metal_name = $this->get_metal_name($symbol_upper);
                    $unit = $this->get_metal_unit($symbol_upper);
                    
                    if (isset($data['timestamp']) && $data['timestamp'] > $latest_timestamp) {
                        $latest_timestamp = $data['timestamp'];
                    }
                ?>
                <tr>
                    <td><?php echo esc_html($metal_name); ?> (<?php echo esc_html($symbol_upper); ?>)</td>
                    <td>
                        <?php if ($show_currency): ?>
                            <span class="currency-symbol"><?php echo esc_html($currency_symbol); ?></span>
                        <?php endif; ?>
                        <?php echo esc_html($formatted_price); ?>
                    </td>
                    <?php if ($show_unit): ?>
                        <td><?php echo esc_html($unit); ?></td>
                    <?php endif; ?>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
        <?php if ($show_updated && $latest_timestamp > 0): ?>
            <div class="metals-updated">Updated: <?php echo esc_html($this->format_timestamp($latest_timestamp)); ?></div>
        <?php endif; ?>
        <?php
        $html = ob_get_clean();
        wp_send_json_success($html);
    }

    function get_metal_name($symbol) {
        $names = [
            'XAU' => 'Gold',
            'XAG' => 'Silver',
            'XPT' => 'Platinum',
            'XPD' => 'Palladium',
        ];
        return isset($names[$symbol]) ? $names[$symbol] : $symbol;
    }

    function get_metal_unit($symbol) {
        $units = [
            'XAU' => 'per oz',
            'XAG' => 'per oz',
            'XPT' => 'per oz',
            'XPD' => 'per oz',
        ];
        return isset($units[$symbol]) ? $units[$symbol] : '';
    }

    function fetch_metal_data($symbol, $base = 'USD') {
        $key = get_option('metalapi_apikey');
        if (empty($key)) {
            return ['success' => false, 'error' => 'API key not configured'];
        }

        $uri = "https://metals-api.com/api/latest?access_key=$key&base=".$base."&symbols=".$symbol;
        $curl = curl_init();
        curl_setopt($curl, CURLOPT_URL, $uri);
        curl_setopt($curl, CURLOPT_HEADER, 0);
        curl_setopt($curl, CURLOPT_SSL_VERIFYPEER, 0);
        curl_setopt($curl, CURLOPT_RETURNTRANSFER, true);
        $response = curl_exec($curl); 
        curl_close($curl);

        if (empty(trim($response))) {
            return ['success' => false, 'error' => 'No response from API'];
        }

        $json = json_decode($response, true);
        return $json;
    }

    function format_price($price, $decimals = 2) {
        return number_format($price, $decimals, '.', ',');
    }

    function format_timestamp($timestamp) {
        $date = new DateTime();
        $date->setTimestamp($timestamp);
        return $date->format('M j, Y') . ' — ' . $date->format('H:i') . ' UTC';
    }

    function render_live_price($attributes) {
        // Validate required parameters
        if (!isset($attributes['symbol']) || empty(trim($attributes['symbol']))) {
            return '<div class="metal-live-price-error" style="padding: 10px; background: #fee; border: 1px solid #fcc; color: #c33; border-radius: 4px;">Error: The "symbol" parameter is required. Example: [metal_live_price symbol="XAU" base="USD" decimals="2" show-unit="true" show-currency="true" show-updated="true" refresh="AUTO"]</div>';
        }
        
        if (!isset($attributes['base']) || empty(trim($attributes['base']))) {
            return '<div class="metal-live-price-error" style="padding: 10px; background: #fee; border: 1px solid #fcc; color: #c33; border-radius: 4px;">Error: The "base" parameter is required. Example: [metal_live_price symbol="XAU" base="USD" decimals="2" show-unit="true" show-currency="true" show-updated="true" refresh="AUTO"]</div>';
        }
        
        if (!isset($attributes['decimals']) || empty(trim($attributes['decimals']))) {
            return '<div class="metal-live-price-error" style="padding: 10px; background: #fee; border: 1px solid #fcc; color: #c33; border-radius: 4px;">Error: The "decimals" parameter is required. Example: [metal_live_price symbol="XAU" base="USD" decimals="2" show-unit="true" show-currency="true" show-updated="true" refresh="AUTO"]</div>';
        }
        
        if (!isset($attributes['show-unit'])) {
            return '<div class="metal-live-price-error" style="padding: 10px; background: #fee; border: 1px solid #fcc; color: #c33; border-radius: 4px;">Error: The "show-unit" parameter is required. Example: [metal_live_price symbol="XAU" base="USD" decimals="2" show-unit="true" show-currency="true" show-updated="true" refresh="AUTO"]</div>';
        }
        
        if (!isset($attributes['show-currency'])) {
            return '<div class="metal-live-price-error" style="padding: 10px; background: #fee; border: 1px solid #fcc; color: #c33; border-radius: 4px;">Error: The "show-currency" parameter is required. Example: [metal_live_price symbol="XAU" base="USD" decimals="2" show-unit="true" show-currency="true" show-updated="true" refresh="AUTO"]</div>';
        }
        
        if (!isset($attributes['show-updated'])) {
            return '<div class="metal-live-price-error" style="padding: 10px; background: #fee; border: 1px solid #fcc; color: #c33; border-radius: 4px;">Error: The "show-updated" parameter is required. Example: [metal_live_price symbol="XAU" base="USD" decimals="2" show-unit="true" show-currency="true" show-updated="true" refresh="AUTO"]</div>';
        }
        
        if (!isset($attributes['refresh']) || empty(trim($attributes['refresh']))) {
            return '<div class="metal-live-price-error" style="padding: 10px; background: #fee; border: 1px solid #fcc; color: #c33; border-radius: 4px;">Error: The "refresh" parameter is required. Example: [metal_live_price symbol="XAU" base="USD" decimals="2" show-unit="true" show-currency="true" show-updated="true" refresh="AUTO"]</div>';
        }
        
        $symbol = strtoupper(trim($attributes['symbol']));
        $base = strtoupper(trim($attributes['base']));
        $decimals = intval($attributes['decimals']);
        $show_unit = $attributes['show-unit'] !== 'false';
        $show_currency = $attributes['show-currency'] !== 'false';
        $show_updated = $attributes['show-updated'] !== 'false';
        $refresh = trim($attributes['refresh']);

        // Validate refresh parameter
        if ($refresh !== 'AUTO') {
            $refresh_int = intval($refresh);
            if ($refresh_int < 1) {
                return '<div class="metal-live-price-error" style="padding: 10px; background: #fee; border: 1px solid #fcc; color: #c33; border-radius: 4px;">Error: Refresh interval must be at least 1 minute. Use "AUTO" for page load refresh, or a number between 1 and 1440 minutes (24 hours).</div>';
            }
            if ($refresh_int > 1440) {
                return '<div class="metal-live-price-error" style="padding: 10px; background: #fee; border: 1px solid #fcc; color: #c33; border-radius: 4px;">Error: Refresh interval cannot exceed 1440 minutes (24 hours). Maximum allowed value is 1440 minutes.</div>';
            }
        }

        $data = $this->fetch_metal_data($symbol, $base);
        
        if (!isset($data['success']) || !$data['success']) {
            $error_msg = isset($data['error']['info']) ? $data['error']['info'] : 'Error fetching data';
            return '<div class="metal-live-price-error">Error: ' . esc_html($error_msg) . '</div>';
        }

        // Try multiple key formats to find the rate
        $price = null;
        $rate_key = $base . $symbol;
        if (isset($data['rates'][$rate_key])) {
            $price = $data['rates'][$rate_key];
        } elseif (isset($data['rates'][$symbol])) {
            // Fallback: try symbol directly
            $price = $data['rates'][$symbol];
        } elseif (isset($data['rates']['USD' . $symbol])) {
            // Fallback: try USD prefix
            $price = $data['rates']['USD' . $symbol];
        }
        
        if ($price === null) {
            return '<div class="metal-live-price-error">Error: Rate not found for ' . esc_html($symbol) . '</div>';
        }
        $formatted_price = $this->format_price($price, $decimals);
        $metal_name = $this->get_metal_name($symbol);
        $unit = $this->get_metal_unit($symbol);
        
        $currency_symbols = [
            'USD' => '$',
            'EUR' => '€',
            'GBP' => '£',
        ];
        $currency_symbol = isset($currency_symbols[$base]) ? $currency_symbols[$base] : $base . ' ';

        $unique_id = 'metal-live-price-' . uniqid();
        $refresh_interval = 0;
        if ($refresh !== 'AUTO') {
            $refresh_interval = intval($refresh);
            if ($refresh_interval < 1 || $refresh_interval > 1440) {
                $refresh_interval = 0;
            } else {
                $refresh_interval = $refresh_interval * 60 * 1000; // Convert to milliseconds
            }
        }

        ob_start();
        ?>
        <div id="<?php echo esc_attr($unique_id); ?>" class="metal-live-price" data-symbol="<?php echo esc_attr($symbol); ?>" data-base="<?php echo esc_attr($base); ?>" data-decimals="<?php echo esc_attr($decimals); ?>" data-show-unit="<?php echo $show_unit ? 'true' : 'false'; ?>" data-show-currency="<?php echo $show_currency ? 'true' : 'false'; ?>" data-show-updated="<?php echo $show_updated ? 'true' : 'false'; ?>" data-refresh="<?php echo esc_attr($refresh); ?>">
            <div class="metal-name"><?php echo esc_html($metal_name); ?> (<?php echo esc_html($symbol); ?>)</div>
            <div class="metal-price">
                <?php if ($show_currency): ?>
                    <span class="currency-symbol"><?php echo esc_html($currency_symbol); ?></span>
                <?php endif; ?>
                <span class="price-value"><?php echo esc_html($formatted_price); ?></span>
                <?php if ($show_unit && !empty($unit)): ?>
                    <span class="price-unit"> / <?php echo esc_html($unit); ?></span>
                <?php endif; ?>
            </div>
            <?php if ($show_updated && isset($data['timestamp'])): ?>
                <div class="metal-updated">Updated: <?php echo esc_html($this->format_timestamp($data['timestamp'])); ?></div>
            <?php endif; ?>
        </div>
        <?php if ($refresh_interval > 0): ?>
        <script>
        (function() {
            var container = document.getElementById('<?php echo esc_js($unique_id); ?>');
            if (!container) return;
            
            function refreshPrice() {
                var xhr = new XMLHttpRequest();
                xhr.open('POST', '<?php echo admin_url('admin-ajax.php'); ?>', true);
                xhr.setRequestHeader('Content-Type', 'application/x-www-form-urlencoded');
                xhr.onreadystatechange = function() {
                    if (xhr.readyState === 4 && xhr.status === 200) {
                        try {
                            var response = JSON.parse(xhr.responseText);
                            if (response.success && response.data) {
                                container.innerHTML = response.data;
                            }
                        } catch(e) {
                            console.error('Error refreshing price:', e);
                        }
                    }
                };
                xhr.send('action=refresh_metal_price&symbol=<?php echo esc_js($symbol); ?>&base=<?php echo esc_js($base); ?>&decimals=<?php echo esc_js($decimals); ?>&show_unit=<?php echo $show_unit ? 'true' : 'false'; ?>&show_currency=<?php echo $show_currency ? 'true' : 'false'; ?>&show_updated=<?php echo $show_updated ? 'true' : 'false'; ?>');
            }
            
            setInterval(refreshPrice, <?php echo $refresh_interval; ?>);
        })();
        </script>
        <?php endif; ?>
        <?php
        return ob_get_clean();
    }

    function render_metals_table($attributes) {
        // Validate required parameters
        if (!isset($attributes['symbols']) || empty(trim($attributes['symbols']))) {
            return '<div class="metals-table-error" style="padding: 10px; background: #fee; border: 1px solid #fcc; color: #c33; border-radius: 4px;">Error: The "symbols" parameter is required. Example: [metals_table symbols="XAU,XAG,XPT,XPD" base="USD" decimals="2" show-unit="true" show-currency="true" show-updated="true" refresh="AUTO"]</div>';
        }
        
        if (!isset($attributes['base']) || empty(trim($attributes['base']))) {
            return '<div class="metals-table-error" style="padding: 10px; background: #fee; border: 1px solid #fcc; color: #c33; border-radius: 4px;">Error: The "base" parameter is required. Example: [metals_table symbols="XAU,XAG,XPT,XPD" base="USD" decimals="2" show-unit="true" show-currency="true" show-updated="true" refresh="AUTO"]</div>';
        }
        
        if (!isset($attributes['decimals']) || empty(trim($attributes['decimals']))) {
            return '<div class="metals-table-error" style="padding: 10px; background: #fee; border: 1px solid #fcc; color: #c33; border-radius: 4px;">Error: The "decimals" parameter is required. Example: [metals_table symbols="XAU,XAG,XPT,XPD" base="USD" decimals="2" show-unit="true" show-currency="true" show-updated="true" refresh="AUTO"]</div>';
        }
        
        if (!isset($attributes['show-unit'])) {
            return '<div class="metals-table-error" style="padding: 10px; background: #fee; border: 1px solid #fcc; color: #c33; border-radius: 4px;">Error: The "show-unit" parameter is required. Example: [metals_table symbols="XAU,XAG,XPT,XPD" base="USD" decimals="2" show-unit="true" show-currency="true" show-updated="true" refresh="AUTO"]</div>';
        }
        
        if (!isset($attributes['show-currency'])) {
            return '<div class="metals-table-error" style="padding: 10px; background: #fee; border: 1px solid #fcc; color: #c33; border-radius: 4px;">Error: The "show-currency" parameter is required. Example: [metals_table symbols="XAU,XAG,XPT,XPD" base="USD" decimals="2" show-unit="true" show-currency="true" show-updated="true" refresh="AUTO"]</div>';
        }
        
        if (!isset($attributes['show-updated'])) {
            return '<div class="metals-table-error" style="padding: 10px; background: #fee; border: 1px solid #fcc; color: #c33; border-radius: 4px;">Error: The "show-updated" parameter is required. Example: [metals_table symbols="XAU,XAG,XPT,XPD" base="USD" decimals="2" show-unit="true" show-currency="true" show-updated="true" refresh="AUTO"]</div>';
        }
        
        if (!isset($attributes['refresh']) || empty(trim($attributes['refresh']))) {
            return '<div class="metals-table-error" style="padding: 10px; background: #fee; border: 1px solid #fcc; color: #c33; border-radius: 4px;">Error: The "refresh" parameter is required. Example: [metals_table symbols="XAU,XAG,XPT,XPD" base="USD" decimals="2" show-unit="true" show-currency="true" show-updated="true" refresh="AUTO"]</div>';
        }
        
        $symbols_str = trim($attributes['symbols']);
        $symbols = array_map('trim', explode(',', $symbols_str));
        $base = strtoupper(trim($attributes['base']));
        $decimals = intval($attributes['decimals']);
        $show_unit = $attributes['show-unit'] !== 'false';
        $show_currency = $attributes['show-currency'] !== 'false';
        $show_updated = $attributes['show-updated'] === 'true';
        $refresh = trim($attributes['refresh']);

        // Validate refresh parameter
        if ($refresh !== 'AUTO') {
            $refresh_int = intval($refresh);
            if ($refresh_int < 1) {
                return '<div class="metals-table-error" style="padding: 10px; background: #fee; border: 1px solid #fcc; color: #c33; border-radius: 4px;">Error: Refresh interval must be at least 1 minute. Use "AUTO" for page load refresh, or a number between 1 and 1440 minutes (24 hours).</div>';
            }
            if ($refresh_int > 1440) {
                return '<div class="metals-table-error" style="padding: 10px; background: #fee; border: 1px solid #fcc; color: #c33; border-radius: 4px;">Error: Refresh interval cannot exceed 1440 minutes (24 hours). Maximum allowed value is 1440 minutes.</div>';
            }
        }

        $currency_symbols = [
            'USD' => '$',
            'EUR' => '€',
            'GBP' => '£',
        ];
        $currency_symbol = isset($currency_symbols[$base]) ? $currency_symbols[$base] : $base . ' ';

        $symbols_str_api = implode(',', array_map('strtoupper', $symbols));
        $data = $this->fetch_metal_data($symbols_str_api, $base);
        
        if (!isset($data['success']) || !$data['success']) {
            $error_msg = isset($data['error']['info']) ? $data['error']['info'] : 'Error fetching data';
            return '<div class="metals-table-error">Error: ' . esc_html($error_msg) . '</div>';
        }

        $unique_id = 'metals-table-' . uniqid();
        $refresh_interval = 0;
        if ($refresh !== 'AUTO') {
            $refresh_interval = intval($refresh);
            if ($refresh_interval < 1 || $refresh_interval > 1440) {
                $refresh_interval = 0;
            } else {
                $refresh_interval = $refresh_interval * 60 * 1000; // Convert to milliseconds
            }
        }

        ob_start();
        ?>
        <div id="<?php echo esc_attr($unique_id); ?>" class="metals-table-wrapper" data-symbols="<?php echo esc_attr($symbols_str); ?>" data-base="<?php echo esc_attr($base); ?>" data-decimals="<?php echo esc_attr($decimals); ?>" data-show-unit="<?php echo $show_unit ? 'true' : 'false'; ?>" data-show-currency="<?php echo $show_currency ? 'true' : 'false'; ?>" data-show-updated="<?php echo $show_updated ? 'true' : 'false'; ?>" data-refresh="<?php echo esc_attr($refresh); ?>">
            <table class="metals-table">
                <thead>
                    <tr>
                        <th>Metal</th>
                        <th>Price</th>
                        <?php if ($show_unit): ?>
                            <th>Unit</th>
                        <?php endif; ?>
                    </tr>
                </thead>
                <tbody>
                    <?php
                    $latest_timestamp = 0;
                    foreach ($symbols as $symbol):
                        $symbol_upper = strtoupper(trim($symbol));
                        
                        // Try multiple key formats to find the rate
                        $price = null;
                        $rate_key = $base . $symbol_upper;
                        if (isset($data['rates'][$rate_key])) {
                            $price = $data['rates'][$rate_key];
                        } elseif (isset($data['rates'][$symbol_upper])) {
                            // Fallback: try symbol directly
                            $price = $data['rates'][$symbol_upper];
                        } elseif (isset($data['rates']['USD' . $symbol_upper])) {
                            // Fallback: try USD prefix
                            $price = $data['rates']['USD' . $symbol_upper];
                        }
                        
                        // Skip if price not found
                        if ($price === null) continue;
                        
                        $formatted_price = $this->format_price($price, $decimals);
                        $metal_name = $this->get_metal_name($symbol_upper);
                        $unit = $this->get_metal_unit($symbol_upper);
                        
                        if (isset($data['timestamp']) && $data['timestamp'] > $latest_timestamp) {
                            $latest_timestamp = $data['timestamp'];
                        }
                    ?>
                    <tr>
                        <td><?php echo esc_html($metal_name); ?> (<?php echo esc_html($symbol_upper); ?>)</td>
                        <td>
                            <?php if ($show_currency): ?>
                                <span class="currency-symbol"><?php echo esc_html($currency_symbol); ?></span>
                            <?php endif; ?>
                            <?php echo esc_html($formatted_price); ?>
                        </td>
                        <?php if ($show_unit): ?>
                            <td><?php echo esc_html($unit); ?></td>
                        <?php endif; ?>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
            <?php if ($show_updated && $latest_timestamp > 0): ?>
                <div class="metals-updated">Updated: <?php echo esc_html($this->format_timestamp($latest_timestamp)); ?></div>
            <?php endif; ?>
        </div>
        <?php if ($refresh_interval > 0): ?>
        <script>
        (function() {
            var container = document.getElementById('<?php echo esc_js($unique_id); ?>');
            if (!container) return;
            
            function refreshTable() {
                var xhr = new XMLHttpRequest();
                xhr.open('POST', '<?php echo admin_url('admin-ajax.php'); ?>', true);
                xhr.setRequestHeader('Content-Type', 'application/x-www-form-urlencoded');
                xhr.onreadystatechange = function() {
                    if (xhr.readyState === 4 && xhr.status === 200) {
                        try {
                            var response = JSON.parse(xhr.responseText);
                            if (response.success && response.data) {
                                container.innerHTML = response.data;
                            }
                        } catch(e) {
                            console.error('Error refreshing table:', e);
                        }
                    }
                };
                xhr.send('action=refresh_metals_table&symbols=<?php echo esc_js($symbols_str); ?>&base=<?php echo esc_js($base); ?>&decimals=<?php echo esc_js($decimals); ?>&show_unit=<?php echo $show_unit ? 'true' : 'false'; ?>&show_currency=<?php echo $show_currency ? 'true' : 'false'; ?>&show_updated=<?php echo $show_updated ? 'true' : 'false'; ?>');
            }
            
            setInterval(refreshTable, <?php echo $refresh_interval; ?>);
        })();
        </script>
        <?php endif; ?>
        <style>
        .metals-table {
            width: 100%;
            border-collapse: collapse;
            margin: 20px 0;
        }
        .metals-table th,
        .metals-table td {
            padding: 12px;
            text-align: left;
            border-bottom: 1px solid #ddd;
        }
        .metals-table th {
            background-color: #f5f5f5;
            font-weight: bold;
        }
        .metals-table tr:hover {
            background-color: #f9f9f9;
        }
        .metals-updated {
            margin-top: 10px;
            font-size: 0.9em;
            color: #666;
        }
        .metal-live-price {
            margin: 20px 0;
        }
        .metal-name {
            font-weight: bold;
            font-size: 1.1em;
            margin-bottom: 5px;
        }
        .metal-price {
            font-size: 1.3em;
            margin: 10px 0;
        }
        .metal-updated {
            font-size: 0.9em;
            color: #666;
            margin-top: 5px;
        }
        </style>
        <?php
        return ob_get_clean();
    }

    function render($attributes){
        $code = get_option('metalapi_000_msg', 'Error: no answer');

        $key = get_option('metalapi_apikey');
        $api = isset($attributes["date"]) && !empty(trim($attributes["date"])) ? date("Y-m-d", strtotime($attributes["date"])) : 'latest';
        $base = !empty(trim($attributes["base"])) ? trim($attributes["base"]) : 'USD';
        $symbol = trim($attributes["symbol"], '""');
        $symbol_user = strtoupper($symbol);

        $uri = "https://metals-api.com/api/$api?access_key=$key&base=".$base."&symbols=".$symbol_user;
        $curl = curl_init();
        curl_setopt($curl, CURLOPT_URL, $uri);
        curl_setopt($curl, CURLOPT_HEADER, 0);
        curl_setopt($curl, CURLOPT_SSL_VERIFYPEER, 0);
        curl_setopt($curl, CURLOPT_RETURNTRANSFER, true);
        $response = curl_exec($curl); 
        curl_close($curl);

        // Debug mejorado - usar parámetro del shortcode
        if (isset($attributes["debug"]) && $attributes["debug"] == "true") {
            echo '<div style="background: #f0f0f0; padding: 10px; margin: 10px; border: 1px solid #ccc;">';
            echo '<strong>DEBUG INFO:</strong><br/>';
            echo 'Símbolo original: ' . $symbol . '<br/>';
            echo 'Símbolo procesado: ' . $symbol_user . '<br/>';
            echo 'URL de petición: ' . $uri . '<br/>';
            echo 'Respuesta: <pre>' . htmlspecialchars($response) . '</pre>';
            echo '</div>';
        }

        if (!empty(trim($response))) {
            $json = json_decode($response, true);
            
            if (isset($json['success']) && $json['success'] == true) { 
                $code = get_option('metalapi_200_msg', '{{base}} {{price}} {{unit}}');
                if (!empty(trim($code))) {
                    if (isset($json["timestamp"])) {
                        $code = str_replace("{{timestamp}}", date($attributes['date-format'], $json["timestamp"]), $code);
                    }
                    
                    if (isset($json["date"])) {
                        $datetime = isset($attributes["date-format"]) && !empty(trim($attributes["date-format"])) ? date($attributes["date-format"], strtotime($json["date"])) : $json["date"];
                        $code = str_replace("{{date}}", $datetime, $code);
                    }

                    $currency_symbols = [
                        'USD' => '$',
                        'EUR' => '€',
                        'GBP' => '£',
                    ];
                    $currency_symbol = isset($currency_symbols[$base]) ? $currency_symbols[$base] : '$';
                    $code = str_replace("{{base}}", $currency_symbol, $code);
                    $code = str_replace("{{currency}}", strtoupper($base), $code);

                    if (isset($json["rates"])) {
                        // Usar el valor correcto: USDXAU en lugar de XAU
                        $priced = 0;
                        if (isset($json["rates"]["USD" . $symbol_user])) {
                            $priced = $json["rates"]["USD" . $symbol_user];
                        } elseif (isset($json["rates"][$symbol_user])) {
                            // Fallback al valor directo si no existe USDXAU
                            $priced = $json["rates"][$symbol_user];
                        }
                        
                        // Debug del precio original
                        if (isset($attributes["debug"]) && $attributes["debug"] == "true") {
                            echo '<div style="background: #e8f5e8; padding: 5px; margin: 5px; border: 1px solid #4caf50;">';
                            echo 'Precio original por onza: ' . $priced . '<br/>';
                            echo 'Valor USDXAU encontrado: ' . (isset($json["rates"]["USD" . $symbol_user]) ? 'SÍ' : 'NO') . '<br/>';
                        }
                        
                        if (!empty($attributes["unit"])) {
                            switch ($attributes["unit"]) {
                                case "kg":
                                    // 1 kg = 35.274 onzas
                                    $priced = $priced * 35.274;
                                    break;
                                case "gram":
                                    // 1 gramo = 0.035274 onzas
                                    $priced = $priced * 0.035274;
                                    break;
                                case "ounce":
                                    $priced = $priced;
                                    break;
                                default:
                                    $priced = $priced;
                                    break;
                            }
                        }
                        
                        // Debug del precio convertido
                        if (isset($attributes["debug"]) && $attributes["debug"] == "true") {
                            echo 'Precio convertido: ' . $priced . '<br/>';
                            echo 'Unidad: ' . ($attributes["unit"] ?? 'ounce') . '<br/>';
                            echo '</div>';
                        }
                        
                        if (isset($attributes["price-round"])) {
                            $round_price = intval(trim($attributes["price-round"], '""'));
                            $priced = round($priced, $round_price);
                        }
                        
                        $separator = isset($attributes["separator"]) ? trim($attributes["separator"]) : '.';
                        $priced = number_format($priced, isset($attributes["price-round"]) ? intval(trim($attributes["price-round"], '""')) : 2, $separator, ',');
                        
                        $code = str_replace('{{price}}', $priced, $code);
                        $code = str_replace("{{symbol}}", $attributes["symbol"], $code);
                    }
                    
                    if (isset($attributes["unit"])) {
                        switch ($attributes["unit"]) {
                            case "kg":
                                $code = str_replace("{{unit}}", 'Kg', $code);
                                break;
                            case "gram":
                                $code = str_replace("{{unit}}", 'Gram', $code);
                                break;
                            case "ounce":
                                $code = str_replace("{{unit}}", 'Ounce', $code);
                                break;
                            default:
                                $code = str_replace("{{unit}}", '', $code);
                                break;
                        }
                    }
                    
                    update_option('metalapi_error', 'Working'); 
                } else {
                    $code = get_option('metalapi_500_msg', 'Error: {{error}}');
                    update_option('metalapi_error', 'stopped');
                    if (isset($json["error"]["info"])) {
                        update_option('metalapi_error', "Error: [".$json["error"]["info"]."]");
                        $code = str_replace("{{error}}", $json["error"]["info"], $code);
                    }
                }
            }
        }
        return $code;
    }
}

$metalsAPI = new MetalsAPI();
add_action('admin_menu', [$metalsAPI, 'addSettings']);
add_shortcode('metals_api_price', [$metalsAPI, 'render']);
add_shortcode('metal_live_price', [$metalsAPI, 'render_live_price']);
add_shortcode('metals_table', [$metalsAPI, 'render_metals_table']);
add_action('wp_ajax_refresh_metal_price', [$metalsAPI, 'ajax_refresh_metal_price']);
add_action('wp_ajax_nopriv_refresh_metal_price', [$metalsAPI, 'ajax_refresh_metal_price']);
add_action('wp_ajax_refresh_metals_table', [$metalsAPI, 'ajax_refresh_metals_table']);
add_action('wp_ajax_nopriv_refresh_metals_table', [$metalsAPI, 'ajax_refresh_metals_table']);
?>